/*
 * Copyright (C) Internet Systems Consortium, Inc. ("ISC")
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef ISC_BASE32_H
#define ISC_BASE32_H 1

/*! \file */

/*
 * Routines for manipulating base 32 and base 32 hex encoded data.
 * Based on RFC 4648.
 *
 * Base 32 hex preserves the sort order of data when it is encoded /
 * decoded.
 *
 * Base 32 hex "np" is base 32 hex but no padding is produced or accepted.
 */

#include <isc/types.h>

/***
 *** Functions
 ***/

isc_result_t
isc_base32hexnp_totext(isc_region_t *source, int wordlength,
		       const char *wordbreak, isc_buffer_t *target);
/*!<
 * \brief Convert data into base32 encoded text.
 *
 * Notes:
 *\li	The base32 encoded text in 'target' will be divided into
 *	words of at most 'wordlength' characters, separated by
 * 	the 'wordbreak' string.  No parentheses will surround
 *	the text.
 *
 * Requires:
 *\li	'source' is a region containing binary data
 *\li	'target' is a text buffer containing available space
 *\li	'wordbreak' points to a null-terminated string of
 *		zero or more whitespace characters
 *
 * Ensures:
 *\li	target will contain the base32 encoded version of the data
 *	in source.  The 'used' pointer in target will be advanced as
 *	necessary.
 */

isc_result_t
isc_base32hexnp_decoderegion(isc_region_t *source, isc_buffer_t *target);
/*!<
 * \brief Decode a packed (no white space permitted) region in
 * base32, base32hex or base32hex non-padded.
 *
 * Requires:
 *\li   'source' is a valid region.
 *\li   'target' is a valid buffer.
 *
 * Returns:
 *\li   #ISC_R_SUCCESS  -- the entire decoded representation of 'cstring'
 *                         fit in 'target'.
 *\li   #ISC_R_BADBASE32 -- 'source' is not a valid base32 encoding.
 */

#endif /* ISC_BASE32_H */
